// Copyright 2018 the V8 project authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef V8_OBJECTS_CALL_SITE_INFO_H_
#define V8_OBJECTS_CALL_SITE_INFO_H_

#include <optional>

#include "src/objects/objects-body-descriptors.h"
#include "src/objects/struct.h"
#include "src/objects/trusted-pointer.h"
#include "torque-generated/bit-fields.h"

// Has to be the last include (doesn't have include guards):
#include "src/objects/object-macros.h"

namespace v8::internal {

class MessageLocation;
class WasmInstanceObject;
class StructBodyDescriptor;

#include "torque-generated/src/objects/call-site-info-tq.inc"

V8_OBJECT class CallSiteInfo : public StructLayout {
 public:
  DEFINE_TORQUE_GENERATED_CALL_SITE_INFO_FLAGS()

#if V8_ENABLE_WEBASSEMBLY
  inline bool IsWasm() const;
  inline bool IsAsmJsWasm() const;
  inline bool IsAsmJsAtNumberConversion() const;
#if V8_ENABLE_DRUMBRAKE
  inline bool IsWasmInterpretedFrame() const;
#endif  // V8_ENABLE_DRUMBRAKE
  inline bool IsBuiltin() const;
#endif  // V8_ENABLE_WEBASSEMBLY

  inline bool IsStrict() const;
  inline bool IsConstructor() const;
  inline bool IsAsync() const;
  bool IsEval() const;
  bool IsUserJavaScript() const;
  bool IsSubjectToDebugging() const;
  bool IsMethodCall() const;
  bool IsToplevel() const;
  bool IsPromiseAll() const;
  bool IsPromiseAllSettled() const;
  bool IsPromiseAny() const;
  bool IsNative() const;

  inline Tagged<HeapObject> code_object(IsolateForSandbox isolate) const;
  inline void set_code_object(Tagged<HeapObject> code,
                              WriteBarrierMode mode = UPDATE_WRITE_BARRIER);

  inline Tagged<JSAny> receiver_or_instance() const;
  inline void set_receiver_or_instance(
      Tagged<JSAny> value, WriteBarrierMode mode = UPDATE_WRITE_BARRIER);

  inline Tagged<Union<JSFunction, Smi>> function() const;
  inline void set_function(Tagged<Union<JSFunction, Smi>> value,
                           WriteBarrierMode mode = UPDATE_WRITE_BARRIER);

  inline int code_offset_or_source_position() const;
  inline void set_code_offset_or_source_position(
      int value, WriteBarrierMode mode = UPDATE_WRITE_BARRIER);

  inline int flags() const;
  inline void set_flags(int value);

  inline Tagged<FixedArray> parameters() const;
  inline void set_parameters(Tagged<FixedArray> value,
                             WriteBarrierMode mode = UPDATE_WRITE_BARRIER);

  // Dispatched behavior.
  DECL_VERIFIER(CallSiteInfo)
  DECL_PRINTER(CallSiteInfo)

  // Used to signal that the requested field is unknown.
  static constexpr int kUnknown = kNoSourcePosition;

  V8_EXPORT_PRIVATE static int GetLineNumber(DirectHandle<CallSiteInfo> info);
  V8_EXPORT_PRIVATE static int GetColumnNumber(DirectHandle<CallSiteInfo> info);

  static int GetEnclosingLineNumber(DirectHandle<CallSiteInfo> info);
  static int GetEnclosingColumnNumber(DirectHandle<CallSiteInfo> info);

  // Returns the script ID if one is attached,
  // Message::kNoScriptIdInfo otherwise.
  static MaybeDirectHandle<Script> GetScript(Isolate* isolate,
                                             DirectHandle<CallSiteInfo> info);
  int GetScriptId() const;
  Tagged<Object> GetScriptName() const;
  Tagged<Object> GetScriptNameOrSourceURL() const;
  Tagged<Object> GetScriptSource() const;
  Tagged<Object> GetScriptSourceMappingURL() const;

  static Handle<PrimitiveHeapObject> GetEvalOrigin(
      DirectHandle<CallSiteInfo> info);
  V8_EXPORT_PRIVATE static DirectHandle<PrimitiveHeapObject> GetFunctionName(
      DirectHandle<CallSiteInfo> info);
  static DirectHandle<String> GetFunctionDebugName(
      DirectHandle<CallSiteInfo> info);
  static DirectHandle<Object> GetMethodName(DirectHandle<CallSiteInfo> info);
  static DirectHandle<String> GetScriptHash(DirectHandle<CallSiteInfo> info);
  static DirectHandle<Object> GetTypeName(DirectHandle<CallSiteInfo> info);

#if V8_ENABLE_WEBASSEMBLY
  // These methods are only valid for Wasm and asm.js Wasm frames.
  uint32_t GetWasmFunctionIndex() const;
  Tagged<WasmInstanceObject> GetWasmInstance() const;
  static DirectHandle<Object> GetWasmModuleName(
      DirectHandle<CallSiteInfo> info);
#endif  // V8_ENABLE_WEBASSEMBLY

  // Returns the 0-based source position, which is the offset into the
  // Script in case of JavaScript and Asm.js, and the wire byte offset
  // in the module in case of actual Wasm. In case of async promise
  // combinator frames, this returns the index of the promise.
  static int GetSourcePosition(DirectHandle<CallSiteInfo> info);

  // Attempts to fill the |location| based on the |info|, and avoids
  // triggering source position table building for JavaScript frames.
  static bool ComputeLocation(DirectHandle<CallSiteInfo> info,
                              MessageLocation* location);

 private:
  static int ComputeSourcePosition(DirectHandle<CallSiteInfo> info, int offset);

  std::optional<Tagged<Script>> GetScript() const;
  Tagged<SharedFunctionInfo> GetSharedFunctionInfo() const;

  friend class Factory;
  friend class TorqueGeneratedCallSiteInfoAsserts;
  friend struct ObjectTraits<CallSiteInfo>;

  TrustedPointerMember<Union<Code, BytecodeArray>, kUnknownIndirectPointerTag>
      code_object_;
  TaggedMember<JSAny> receiver_or_instance_;
  TaggedMember<Union<JSFunction, Smi>> function_;
  TaggedMember<Smi> code_offset_or_source_position_;
  TaggedMember<Smi> flags_;
  TaggedMember<FixedArray> parameters_;
} V8_OBJECT_END;

template <>
struct ObjectTraits<CallSiteInfo> {
  using BodyDescriptor = StackedBodyDescriptor<
      FixedBodyDescriptor<offsetof(CallSiteInfo, receiver_or_instance_),
                          sizeof(CallSiteInfo), sizeof(CallSiteInfo)>,
      WithStrongTrustedPointer<offsetof(CallSiteInfo, code_object_),
                               kUnknownIndirectPointerTag>>;
};

class IncrementalStringBuilder;
void SerializeCallSiteInfo(Isolate* isolate, DirectHandle<CallSiteInfo> frame,
                           IncrementalStringBuilder* builder);
V8_EXPORT_PRIVATE
MaybeDirectHandle<String> SerializeCallSiteInfo(
    Isolate* isolate, DirectHandle<CallSiteInfo> frame);

}  // namespace v8::internal

#include "src/objects/object-macros-undef.h"

#endif  // V8_OBJECTS_CALL_SITE_INFO_H_
