// Copyright 2011 the V8 project authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef V8_HEAP_SPACES_H_
#define V8_HEAP_SPACES_H_

#include <atomic>
#include <memory>

#include "src/base/iterator.h"
#include "src/base/macros.h"
#include "src/common/globals.h"
#include "src/heap/allocation-observer.h"
#include "src/heap/base-page.h"
#include "src/heap/base-space.h"
#include "src/heap/base/active-system-pages.h"
#include "src/heap/free-list.h"
#include "src/heap/linear-allocation-area.h"
#include "src/heap/list.h"
#include "src/heap/main-allocator.h"
#include "src/heap/memory-chunk-layout.h"
#include "src/heap/mutable-page.h"
#include "src/heap/normal-page.h"
#include "src/heap/slot-set.h"
#include "src/objects/objects.h"
#include "src/utils/allocation.h"
#include "src/utils/utils.h"
#include "testing/gtest/include/gtest/gtest_prod.h"  // nogncheck

namespace v8 {
namespace internal {

namespace heap {
class HeapTester;
class TestCodePageAllocatorScope;
}  // namespace heap

class AllocationObserver;
class FreeList;
class Heap;
class Isolate;
class LargeObjectSpace;
class LargePage;
class ObjectIterator;
class PagedSpaceBase;
class SemiSpace;

// Some assertion macros used in the debugging mode.

#define DCHECK_VALID_REGULAR_OBJECT_SIZE(size) \
  DCHECK((0 < size) && (size <= kMaxRegularHeapObjectSize))

#define DCHECK_VALID_REGULAR_CODEOBJECT_SIZE(size) \
  DCHECK((0 < size) && (size <= MemoryChunkLayout::MaxRegularCodeObjectSize()))

template <typename Enum, typename Callback>
void ForAll(Callback callback) {
  for (int i = 0; i < static_cast<int>(Enum::kNumValues); i++) {
    callback(static_cast<Enum>(i), i);
  }
}

// ----------------------------------------------------------------------------
// Space is the abstract superclass for all allocation spaces that are not
// sealed after startup (i.e. not ReadOnlySpace).
class V8_EXPORT_PRIVATE Space : public BaseSpace {
 public:
  Space(Heap* heap, AllocationSpace id, std::unique_ptr<FreeList> free_list)
      : BaseSpace(heap, id), free_list_(std::move(free_list)) {}

  ~Space() override = default;

  Space(const Space&) = delete;
  Space& operator=(const Space&) = delete;

  // Returns size of objects. Can differ from the allocated size
  // (e.g. see OldLargeObjectSpace).
  virtual size_t SizeOfObjects() const { return Size(); }

  // Return the available bytes without growing.
  virtual size_t Available() const = 0;

  virtual std::unique_ptr<ObjectIterator> GetObjectIterator(Heap* heap) = 0;

  virtual MutablePage* first_page() { return memory_chunk_list_.front(); }
  virtual MutablePage* last_page() { return memory_chunk_list_.back(); }

  virtual const MutablePage* first_page() const {
    return memory_chunk_list_.front();
  }
  virtual const MutablePage* last_page() const {
    return memory_chunk_list_.back();
  }

  virtual heap::List<MutablePage>& memory_chunk_list() {
    return memory_chunk_list_;
  }

  virtual NormalPage* InitializePage(MutablePage* chunk) { UNREACHABLE(); }

  virtual void NotifyBlackAreaCreated(size_t size) {}
  virtual void NotifyBlackAreaDestroyed(size_t size) {}

  FreeList* free_list() { return free_list_.get(); }

  Address FirstPageAddress() const {
    DCHECK_NOT_NULL(first_page());
    return first_page()->ChunkAddress();
  }

#ifdef DEBUG
  virtual void Print() = 0;
#endif

 protected:
  // The List manages the pages that belong to the given space.
  heap::List<MutablePage> memory_chunk_list_;
  std::unique_ptr<FreeList> free_list_;
};

static_assert(sizeof(std::atomic<intptr_t>) == kSystemPointerSize);

// -----------------------------------------------------------------------------
// Interface for heap object iterator to be implemented by all object space
// object iterators.

class V8_EXPORT_PRIVATE ObjectIterator : public Malloced {
 public:
  // Note: The destructor can not be marked as `= default` as this causes
  // the compiler on C++20 to define it as `constexpr` resulting in the
  // compiler producing warnings about undefined inlines for Next()
  // on classes inheriting from it.
  virtual ~ObjectIterator() {}
  virtual Tagged<HeapObject> Next() = 0;
};

template <class PageType>
class PageIteratorImpl
    : public base::iterator<std::forward_iterator_tag, PageType> {
 public:
  explicit PageIteratorImpl(PageType* p) : p_(p) {}
  PageIteratorImpl(const PageIteratorImpl&) V8_NOEXCEPT = default;
  PageIteratorImpl& operator=(const PageIteratorImpl&) V8_NOEXCEPT = default;

  PageType* operator*() { return p_; }
  bool operator==(const PageIteratorImpl<PageType>& rhs) const {
    return rhs.p_ == p_;
  }
  bool operator!=(const PageIteratorImpl<PageType>& rhs) const {
    return rhs.p_ != p_;
  }
  inline PageIteratorImpl<PageType>& operator++();
  inline PageIteratorImpl<PageType> operator++(int);

 private:
  PageType* p_;
};

using PageIterator = PageIteratorImpl<NormalPage>;
using ConstPageIterator = PageIteratorImpl<const NormalPage>;
using LargePageIterator = PageIteratorImpl<LargePage>;
using ConstLargePageIterator = PageIteratorImpl<const LargePage>;

class PageRange {
 public:
  using iterator = PageIterator;
  PageRange(NormalPage* begin, NormalPage* end) : begin_(begin), end_(end) {}
  inline explicit PageRange(NormalPage* page);

  iterator begin() { return iterator(begin_); }
  iterator end() { return iterator(end_); }

 private:
  NormalPage* begin_;
  NormalPage* end_;
};

class ConstPageRange {
 public:
  using iterator = ConstPageIterator;
  ConstPageRange(const NormalPage* begin, const NormalPage* end)
      : begin_(begin), end_(end) {}
  inline explicit ConstPageRange(const NormalPage* page);

  iterator begin() { return iterator(begin_); }
  iterator end() { return iterator(end_); }

 private:
  const NormalPage* begin_;
  const NormalPage* end_;
};

class V8_EXPORT_PRIVATE SpaceWithLinearArea : public Space {
 public:
  // Creates this space and uses the existing `allocator`. It doesn't create a
  // new MainAllocator instance.
  SpaceWithLinearArea(Heap* heap, AllocationSpace id,
                      std::unique_ptr<FreeList> free_list);

  virtual AllocatorPolicy* CreateAllocatorPolicy(MainAllocator* allocator) = 0;

  friend class MainAllocator;
};

class V8_EXPORT_PRIVATE SpaceIterator : public Malloced {
 public:
  explicit SpaceIterator(Heap* heap);
  virtual ~SpaceIterator();

  bool HasNext();
  Space* Next();

 private:
  Heap* heap_;
  int current_space_;  // from enum AllocationSpace.
};

// Iterates over all memory chunks in the heap (across all spaces).
class MemoryChunkIterator {
 public:
  explicit MemoryChunkIterator(Heap* heap) : space_iterator_(heap) {}

  V8_INLINE bool HasNext();
  V8_INLINE MutablePage* Next();

 private:
  SpaceIterator space_iterator_;
  MutablePage* current_chunk_ = nullptr;
};

}  // namespace internal
}  // namespace v8

#endif  // V8_HEAP_SPACES_H_
